
package kochModel.factory;

import kochModel.*;

import lrs.*;
import java.awt.*;
import java.awt.geom.*;

import kochModel.*;

/**
 * A factory that makes Koch curves that grow from a single line segment.  This type of factory contains
 * a list of prototype points from a pattern scaled to the unit line segment from (0,0) to (1,0).   A MakeMAnyAlgo object
 * is used to process the prototype list, given an arbitrary line segment in 2D space and produce
 * an Affine transformed (rotated, scaled and translated) version of the prototype pattern, i..e. the pattern
 * is mapped onto the line segment and replaces it..
 * @dependency kochModel.factory.AffineData instantiates
 */
public class CurveFactory implements IFactory {
 /**
  * The prototype list of double precision point objects (class Point2D.Double).
  * This list specifies the non-endpoints of the poly-line  that all straight lines will be replaced with when the curve grows.
  * The prototype is defined as having endpoints at (0.0, 0.0) and (1.0, 0.0), i.e. it is defined over the unit distance.
  * The prototype list does not contain the endpoints.   The first point is the first node in the list.
  * The poly line is assumed to be continuous, i.e. no gaps.
   * Note:  When creating the prototype list, insert the points in backwards order so the will end up with the first point at the top of the list.
  */
 private LRStruct<Point2D.Double> protoList = new LRStruct<Point2D.Double>();
 /**
  * The IAlgo that is used to create the list of  base case Koch curves from the prototype list.
  */
 private MakeManyAlgo makeManyAlgo = new MakeManyAlgo();
 /**
  * Creates a single base case Koch curve.  Used to generate the initial Koch curve.
  * @param a The first endpoint of the straight line.
  * @param b The second endpoint of the straight line.
  */
 public Koch makeBase(Point2D.Double a, Point2D.Double b) {
  return new Koch(a, b);
 }

 /**
  * Constructs the list of base-case Koch curves from the prototype list and a pair of endpoints.  Returns a Koch curve in a InductState that uses the new list of Koch curves.
  * @param a The first endpoint.
  * @param b The second endpoint.
  */
 public Koch makeInduct(Point2D.Double a, Point2D.Double b) {
  LRStruct<Koch> kochList = protoList.execute(makeManyAlgo, new AffineData(a, b));
    return new Koch(a, b, kochList);
 }

  public LRStruct<Point2D.Double> getProtoList() {
    return protoList;
  }
}

